<?php
namespace model;

class Game extends \Pawn\Model
{
	// Migration
	public static function migration()
	{
		return array
		(
			array('CREATE', array
			(
				'name'					=>	'VARCHAR(64)',
				'tag'					=>	'CHAR(16) UNIQUE',
				'description'			=>	'TEXT',
				'slug'					=>	'VARCHAR(255)',
				'width'					=>	'INT',
				'height'				=>	'INT',
				'thumbnail_url'			=>	'VARCHAR(255)',
				'swf_url'				=>	'VARCHAR(255)',
				'category_id'			=>	'INT UNSIGNED',
				'categories'			=>	'VARCHAR(255)',
				'keywords'				=>	'TEXT',
				'leaderboard_enabled'	=>	'TINYINT',
				
				'featured'				=>	'TINYINT DEFAULT 0',
				'plays'					=>	'INT UNSIGNED DEFAULT 0',
				'comments'				=>	'INT UNSIGNED DEFAULT 0',
				'favourites'			=>	'INT UNSIGNED DEFAULT 0',
				'rating'				=>	'FLOAT DEFAULT 0'
			)),
			array('CONFIG', array
			(
				'publisher_id'			=>	'',
				'publisher_secret_key'	=>	''
			)),
			array('CONFIG', array
			(
				'verification_code'		=>	'',
				'host_automatically'	=>	'0'
			))
		);
	}
	
	// Identifier
	public function __toString()
	{
		return $this->name;
	}
	
	// Administration
	const sortColumn = 'ts_edit';
	const sortAscending = 0;
	
	public function listing()
	{
		return array
		(
			'name'			=>	$this->__toString(),
			'featured'		=>	$this->img_enabled($this->featured),
			'category_id'	=>	\model\Category::get('WHERE id=?', $this->category_id, false)->fetch()->__toString(),
			'ts_edit'		=>	\Pawn::get_date($this->ts_edit)
		);
	}
	
	public static function actions()
	{
		return array('add', 'fetch');
	}
	
	public static function bulk()
	{
		return array
		(
			'trash'		=>	true,
			'edit'		=>	false,
			'feature'	=>	false
		);
	}
	
	public static function config()
	{
		return array
		(
			'publisher_id'			=>	array('text', array('optional'=>true, 'length'=>16)),
			'publisher_secret_key'	=>	array('text', array('optional'=>true, 'length'=>32)),
			'verification_code'		=>	array('text', array('optional'=>true)),
			'host_automatically'	=>	array('checkbox')
		);
	}
	
	// Fetch
	public static function fetch($controller)
	{
		$form = new \lib\Form('fetch');
		
		$form->field('limit', 'text', array('positive_integer'=>true), 10);
		$form->field('offset', 'text', array('non_negative_integer'=>true), 0);
		$form->field('recommended', 'checkbox', true);
		$form->field('leaderboard', 'checkbox', true);
		
		if($form->validate())
		{
			$url = 'http://feedmonger.mochimedia.com/feeds/query/?q=not%20tags%3azh-cn';
			
			if($form->request('recommended'))
			{
				$url .= '%20and%20recommendation%3A%3E%3D4';
			}
			
			if($form->request('leaderboard'))
			{
				$url .= '%20and%20leaderboard_enabled';
			}
			
			$url .= '&offset='.$form->request('offset').'&limit='.$form->request('limit');
			
			$contents = \Pawn::remote_contents($url);
			
			$request = json_decode($contents);
			
			foreach($request->games as $game)
			{
				$instance = new Game(array('obj'=>$game), true);
				
				$instance->create();
			}
			
			return $controller->redirect('Pawn', 'model', array('name'=>'Game'));
		}
		
		return $controller->render('games/fetch.html', array
		(
			'form'			=>	$form,
			
			'navigation'	=>	$controller->navigation('Game')
		));
	}
	
	// Validation
	public function columns($preset)
	{
		if($preset == 'administrate')
		{
			$categories = array();
			
			$result = \model\Category::get()->fetchAll();
			
			foreach($result as $category)
			{
				$categories[$category->id] = $category->__toString();
			}
			
			return array
			(
				'name'			=>	array('text', array()),
				'description'	=>	array('textarea', array()),
				'thumbnail_url'	=>	array('file', array('file_type'=>'image', 'image_size'=>array(100, 100))),
				'swf_url'		=>	array('file', array('file_type'=>'swf'), $this->swf_url, 'swf'),
				'width'			=>	array('text', array('positive_integer'=>true)),
				'height'		=>	array('text', array('positive_integer'=>true)),
				'category_id'	=>	array('select', $categories, $this->category_id, array(), false),
				'keywords'		=>	array('text', array('optional'=>true))
			);
		}
		
		return array
		(
			'obj'	=>	array('FieldObject', array('c_unique_game'=>array($this, 'unique_game')))
		);
	}
	
	public function unique_game($object)
	{
		return static::get('WHERE tag=?', $object->game_tag, false)->fetch() === false;
	}
	
	public function fields($action, $fields, $preset)
	{
		if($preset == 'administrate')
		{
			$save = array
			(
				'name'			=>	$fields['name'],
				'description'	=>	$fields['description'],
				'thumbnail_url'	=>	$fields['thumbnail_url'],
				'swf_url'		=>	$fields['swf_url'],
				'width'			=>	$fields['width'],
				'height'		=>	$fields['height'],
				'category_id'	=>	$fields['category_id'],
				'categories'	=>	$fields['category_id'],
				'keywords'		=>	$fields['keywords']
			);
			
			if($action == 'create')
			{
				$save['tag'] = '________________';
				$save['slug'] = '';
				$save['leaderboard_enabled'] = 0;
			}
			
			return $save;
		} else
		{
			$category = \model\Category::get('WHERE name=?', $fields['obj']->category)->fetch();
			
			$categories = array();
			
			foreach($fields['obj']->categories as $name)
			{
				$instance = \model\Category::get('WHERE name=?', $name)->fetch();
				
				$categories[] = $instance->id;
			}
			
			if(!$category)
			{
				return false;
			}
			
			return array
			(
				'name'					=>	$fields['obj']->name,
				'tag'					=>	$fields['obj']->game_tag,
				'description'			=>	$fields['obj']->description,
				'slug'					=>	$fields['obj']->slug,
				'width'					=>	$fields['obj']->width,
				'height'				=>	$fields['obj']->height,
				'thumbnail_url'			=>	$fields['obj']->thumbnail_url,
				'swf_url'				=>	$fields['obj']->swf_url,
				'category_id'			=>	$category->id,
				'categories'			=>	implode(',', $categories),
				'keywords'				=>	implode(',', $fields['obj']->tags),
				'leaderboard_enabled'	=>	$fields['obj']->leaderboard_enabled?1:0
			);
		}
	}
	
	// Create
	public function create($name='instance', $preset='default')
	{
		$form = parent::create($name, $preset);
		
		if($form->valid)
		{
			$this->category->games++;
			
			if($preset == 'administrate')
			{
				$this->tag = str_pad($this->id, 16, '_');
				$this->slug = $this->id.'-'.preg_replace('/[^A-Za-z0-9-]/', '', str_replace(' ', '-', strtolower($this->name)));
			} else
			{
				if((bool) \Pawn::setting('Game.host_automatically'))
				{
					// Host swf
					$content = \Pawn::remote_contents($this->swf_url);
					$this->swf_url = $this->id.'.swf';
					
					file_put_contents($this->filePath('swf_url').$this->swf_url, $content);
					
					// Host thumbnail
					$content = \Pawn::remote_contents($this->thumbnail_url);
					$this->thumbnail_url = $this->id.'.jpg';
					
					file_put_contents($this->filePath('thumbnail_url').$this->thumbnail_url, $content);
				}
				
				$control_scheme = json_decode($this->obj->control_scheme);
				
				foreach($control_scheme as $action=>$key)
				{
					$instance = new \model\Control(array
					(
						'game_id'	=>	$this->id,
						'action'	=>	$action,
						'key'		=>	$key
					), true);
					
					$instance->create();
				}
			}
		}
		
		return $form;
	}
	
	// Trash
	public function trash()
	{
		$this->category->games--;
		
		\model\Feature::get('WHERE game_id=?', $this->id, false, 'delete');
		\model\Favourite::get('WHERE game_id=?', $this->id, false, 'delete');
		
		$this->favourites = 0;
		
		parent::trash();
	}
	
	// Restore
	public function restore()
	{
		$this->category->games++;
		
		parent::restore();
	}
	
	// Delete
	public function delete()
	{
		\model\Control::get('WHERE game_id=?', $this->id, false, 'delete');
		\model\Comment::get('WHERE target="game" AND target_id=?', $this->id, false, 'delete');
		
		parent::delete();
	}
	
	// Feature
	public function feature()
	{
		if($this->featured)
		{
			$this->featured = 0;
			
			\model\Feature::get('WHERE game_id=?', $this->id, false, 'delete');
		} else
		{
			$this->featured = 1;
			
			$instance = new \model\Feature(array
			(
				'game_id'	=>	$this->id
			), true);
			
			$instance->create();
		}
	}
	
	// Rank
	public function rank()
	{
		$user_id = \lib\Session::user()->id;
		
		$submissions = \model\Score::get('WHERE game_id=? AND user_id=? LIMIT 1', array($this->id, $user_id))->fetchAll();
		
		if(!$submissions)
		{
			return '';
		}
		
		$score = $submissions[0]->score;
		$order = $submissions[0]->order;
		
		foreach($submissions as $submission)
		{
			if(($order == 'asc' && $submission->score < $score) || ($order == 'desc' && $submission->score > $score))
			{
				$score = $submission->score;
			}
		}
		
		return \model\Score::num_rows('WHERE game_id=? AND user_id!=? GROUP BY user_id HAVING '.(($order == 'asc')?'MIN(score)<?':'MAX(score)>?'), array($this->id, $user_id, $score)) + 1;
	}
	
	// Total games
	public static function total_games()
	{
		return \lib\db::query('SELECT SUM(games) FROM '.PREFIX.'category WHERE enabled=1')->fetchColumn();
	}
}